#! /bin/bash

# All top-level commands here must run their internal commands in a subshell so that exiting doesn't kill the main shell

tines-console() {
  (
    set -euo pipefail
    sudo docker-compose run --rm --name=tines-console tines-app bundle exec rails console -- --noautocomplete
  )
}

_tines-resolve-container-id() {
  set -euo pipefail
  local CONTAINER_ID
  local service_name=${1:-tines-app}
  CONTAINER_ID=$(sudo docker ps -f "name=$service_name" -q | tail -n1)

  if [ -z "$CONTAINER_ID" ]; then
    echo "No container found for '$service_name' service when running 'docker ps'."
    exit 1
  fi

  echo "$CONTAINER_ID"
}

_tines-resolve-sidekiq-container-ids() {
  set -euo pipefail
  local SIDEKIQ_CONTAINER_IDS=()
  SIDEKIQ_CONTAINER_IDS=($(sudo docker ps -f name=tines-sidekiq -q))

  if [ ${#SIDEKIQ_CONTAINER_IDS[@]} -eq 0 ]; then
    echo "No containers found for 'tines-sidekiq' service when running 'docker ps'."
    exit 1
  fi

  echo "${SIDEKIQ_CONTAINER_IDS[@]}"
}

_tines-app-command() {
  # Open a subshell so that exiting doesn't kill main shell
  CONTAINER_ID=$(_tines-resolve-container-id tines-app)
  sudo docker exec -it "$CONTAINER_ID" "$@"
}

tines-app-bundle-exec() {
  _tines-app-command bundle exec "$@"
}

tines-app-console() {
  tines-app-bundle-exec rails console -- --noautocomplete
}

tines-app-shell() {
  _tines-app-command bash
}

tines-sidekiq-bundle-exec() {
  _tines-docker-exec tines-sidekiq bundle exec "$@"
}

tines-sidekiq-console() {
  CONTAINER_ID=$(_tines-resolve-container-id "$1")
  _tines-docker-exec "$CONTAINER_ID" bundle exec rails console -- --noautocomplete
}

tines-sidekiq-shell() {
  CONTAINER_ID=$(_tines-resolve-container-id "$1")
  _tines-docker-exec "$CONTAINER_ID" bash
}

_tines-docker-exec() {
  (
    set -euo pipefail
    sudo docker exec -it "$1" "${@:2}"
  )
}

tines-postgres-console() {
  _tines-docker-exec postgres psql -U tines -d tines_production
}

tines-postgres-shell() {
  _tines-docker-exec postgres bash
}

tines-redis-console() {
  _tines-docker-exec redis redis-cli -n 1
}

tines-redis-shell() {
  _tines-docker-exec redis sh
}

tines-nginx-shell() {
  _tines-docker-exec nginx bash
}

_tines-ensure-container() {
  printf "Checking %s..." "$1"
  CONTAINER_ID=$(_tines-resolve-container-id "$1")
  printf " Success\n"
}

tines-troubleshoot() {
  (
    set -euo pipefail

    echo "Ensuring containers exist for Tines services..."
    echo
    _tines-ensure-container tines-app
    _tines-ensure-container tines-sidekiq
    _tines-ensure-container nginx
    _tines-ensure-container postgres
    _tines-ensure-container redis
    echo

    echo "tines-app health check results:"
    echo
    echo $(_tines-app-command curl -k https://nginx/health)
  )
}

_tines-logs() {
  (
    echo "Printing logs for $@"
    echo
    set -euo pipefail
    sudo docker logs $@
    echo
  )
}

tines-app-logs() {
  CONTAINER_ID=$(_tines-resolve-container-id tines-app)
  echo "Printing logs for tines-app"
  echo
  _tines-logs "$CONTAINER_ID"
  echo
}

tines-app-logs-follow() {
  CONTAINER_ID=$(_tines-resolve-container-id tines-app)
  echo "Printing logs for tines-app"
  echo
  _tines-logs -f "$CONTAINER_ID"
  echo
}

tines-sidekiq-logs() {
  local service_name=${1:-tines-sidekiq}
  CONTAINER_ID=$(_tines-resolve-container-id "$service_name")

  echo "Printing logs for $service_name"
  echo
  _tines-logs "$CONTAINER_ID"
  echo
}

tines-sidekiq-logs-follow() {
  local service_name=${1:-tines-sidekiq}
  CONTAINER_ID=$(_tines-resolve-container-id "$service_name")

  _tines-logs -f "$CONTAINER_ID"
}

tines-nginx-logs() {
  _tines-logs nginx
}

tines-nginx-logs-follow() {
  _tines-logs -f nginx
}

tines-logs() {
  tines-app-logs

  # Print out tines-sidekiq logs for all running tines-sidekiq containers.
  local SIDEKIQ_CONTAINER_IDS=($(_tines-resolve-sidekiq-container-ids))

  if [ ${#SIDEKIQ_CONTAINER_IDS[@]} -eq 0 ]; then
    echo "No containers found for 'tines-sidekiq' service when running 'docker ps'."
    exit 1
  fi

  for container_id in "${SIDEKIQ_CONTAINER_IDS[@]}"; do
    tines-sidekiq-logs "$container_id"
  done

  tines-nginx-logs
}

_tines-inspect() {
  (
    echo "Inspecting $1"
    echo
    set -euo pipefail
    sudo docker inspect "$1"
    echo
  )
}

tines-app-inspect() {
  CONTAINER_ID=$(_tines-resolve-container-id tines-app)
  _tines-inspect "$CONTAINER_ID"
}

# Inspect all running tines-sidekiq containers.
tines-sidekiq-inspect() {
  local SIDEKIQ_CONTAINER_IDS=($(_tines-resolve-sidekiq-container-ids))

  if [ ${#SIDEKIQ_CONTAINER_IDS[@]} -eq 0 ]; then
    echo "No containers found for 'tines-sidekiq' service when running 'docker ps'."
    exit 1
  fi

  for container_id in "${SIDEKIQ_CONTAINER_IDS[@]}"; do
    _tines-inspect "$container_id"
  done
}

tines-nginx-inspect() {
  _tines-inspect nginx
}

tines-inspect() {
  tines-app-inspect
  tines-sidekiq-inspect
  tines-nginx-inspect
}

_ensure-directory-and-du() {
  [ -d "$1" ] && sudo du -sh "$1" || "Error: Directory $1 does not exist."
}

tines-troubleshoot-diskspace() {
  (
    set -uo pipefail

    echo "Checking diskspace..."
    echo

    echo "Checking all containers and their sizes"
    echo
    sudo docker ps -as
    echo

    echo "Checking available diskspace"
    echo
    sudo df -h
    echo

    echo "Checking space taken up by tines-backup"
    echo
    _ensure-directory-and-du "/opt/tines/db-backup/"
    echo

    echo "Checking space allocated to postgres and redis"
    echo
    _ensure-directory-and-du "/var/lib/docker/volumes"
    echo

    echo "Checking space used up by psql tables"
    echo
    _tines-docker-exec postgres psql -U tines -d tines_production -c '\dt+;'
    echo

    echo "Checking space used up by logs"
    echo
    _ensure-directory-and-du "/var/lib/docker/containers"
    echo

    echo "Done checking diskspace"
    echo
  )
}

tines-memory-footprint() {
  (
    echo "Memory footprint per process"
    echo
    sudo ps aux | head -1
    ps aux | sort -rnk 4 | head -50
    echo
  )
}

_tines_reload_nginx() {
  sudo docker exec nginx /usr/sbin/nginx -s reload
}

tines-restart() {
  (
    set -euo pipefail

    # This code is copied in upgrade.sh, the 2 should be kept similar
    local APP_SERVICE_NAME=tines-app
    local old_container_id
    old_container_id=$(sudo docker ps -f name=$APP_SERVICE_NAME -q | tail -n1)
    echo "Loading new app container"
    sudo docker-compose up -d --no-deps --scale $APP_SERVICE_NAME=2 --no-recreate $APP_SERVICE_NAME

    local new_container_id
    new_container_id=$(sudo docker ps -f name=$APP_SERVICE_NAME -q | head -n1)
    local new_container_ip
    new_container_ip=$(sudo docker inspect -f '{{range.NetworkSettings.Networks}}{{.IPAddress}}{{end}}' "$new_container_id")
    echo
    echo "Waiting for new container"
    echo
    if ! curl --silent --retry-connrefused --retry 60 --retry-delay 1 --fail http://"$new_container_ip":3000/is_up; then
      echo "New tines-app container did not start successfully."
      exit 1
    fi
    echo
    echo "New container is up"
    echo
    _tines_reload_nginx
    echo "Stopping old app container"
    sudo docker stop "$old_container_id" && sudo docker rm "$old_container_id"
    sudo docker-compose up -d --no-deps --scale $APP_SERVICE_NAME=1 --no-recreate $APP_SERVICE_NAME
    _tines_reload_nginx
    echo "Restarting other containers"
    sudo docker-compose up --timeout 30 -d --remove-orphans
  )
}

tines-update-tenant-limits() {
  (
    set -euo pipefail
    echo "Please enter new agent limit "
    read agentLimit

    echo "Please enter new daily event limit "
    read eventLimit

    echo "Please enter new user limit "
    read userLimit

    echo "Please enter new story limit "
    read storyLimit

    tines-app-bundle-exec rake tines:update_limits\["$agentLimit","$eventLimit","$userLimit","$storyLimit"\]
  )
}

tines-update-tenant-name() {
  (
    set -euo pipefail
    echo "Please enter new tenant name "
    read name

    tines-app-bundle-exec rake tines:update_name\["$name"\]
  )
}

tines-update-tenant-domain() {
  (
    set -euo pipefail
    echo "Please enter new tenant domain "
    read domain

    tines-app-bundle-exec rake tines:update_domain\["$domain"\]
  )
}

tines-tenant-info() {
  (
    set -euo pipefail
    tines-app-bundle-exec rake tines:print_tenant_info
  )
}

echo "Loaded Tines commands in shell"
