#!/bin/bash

set -e

LOG_DIR="$(pwd)/log/setup-logs"
LOG_FILE="$LOG_DIR/setup_$(date +%Y-%m-%d_%H-%M-%S).log"
mkdir -p "$LOG_DIR"

log_cmd() {
  "$@" 2>&1 | tee -a "$LOG_FILE"
}

if docker compose version &> /dev/null; then
  docker_compose=(docker compose)
else
  docker_compose=(docker-compose)
fi

if [[ "${USE_OLD_DOCKER_COMPOSE_SHELL_COMMAND:-false}" == "true" ]]; then
  log_cmd echo "Using docker-compose as shell command"
  docker_compose=(docker-compose)
fi

function setup_tines_user {
  if grep -qi 'amazon' /etc/os-release; then
    log_cmd sudo adduser --comment 'Tines' --uid 8463 tines
  else
    log_cmd sudo adduser --disabled-login --gecos 'Tines' --uid 8463 tines
  fi

  export CONDITIONAL_LOCAL_TINES_USER="8463:8463"
  export CONDITIONAL_MAPPED_RO_LOCAL_ETC_PASSWD_FILE="/etc/passwd:/etc/passwd:ro"
  export CONDITIONAL_MAPPED_RO_LOCAL_ETC_GROUP_FILE="/etc/group:/etc/group:ro"

  "${docker_compose[@]}" up --no-start db redis

  postgres_vol_location=$(docker inspect postgres --format "{{range (.Mounts)}}{{println .Source}}{{end}}" | grep data)

  redis_vol_location=$(docker inspect redis --format "{{range (.Mounts)}}{{println .Source}}{{end}}" | grep data)

  sudo chown -R tines:tines "$postgres_vol_location"
  sudo chown -R tines:tines "$redis_vol_location"
}

log_cmd echo "===== Docker status:"
log_cmd docker -v
log_cmd "${docker_compose[@]}" version
log_cmd echo
log_cmd echo "===== Running containers:"
log_cmd docker ps -a
log_cmd echo
log_cmd echo "===== Docker volumes:"
log_cmd docker volume ls
log_cmd echo
log_cmd echo "Current directory: $(pwd)"
log_cmd echo

if [ ! -f "docker-compose.yml" ] && [ ! -f "docker-compose.yaml" ]; then
  log_cmd echo "No docker-compose.yml file found in this directory. You'll need to run this script from the directory you extracted the release ZIP file to."
  exit 1
fi

if [ ! -f ".env" ]; then
  log_cmd echo "No .env file found in this directory. You'll need to create your .env file before continuing with the setup."
  exit 1
fi

if [ ! -f "tines.key" ] || [ ! -f "tines.crt" ]; then
  log_cmd echo "SSL cert files (tines.key & tines.crt) were not found in this directory. You'll need to create these files before continuing with the setup."
  exit 1
fi

if [[ $(pwd) != "/opt/tines" ]]; then
  read -p "We recommend installing Tines in /opt/tines. Are you sure you want to continue in $(pwd)? <Y/N>" prompt
  if [[ $prompt != "y" && $prompt != "Y" && $prompt != "yes" && $prompt != "Yes" ]]; then
    echo
    exit 0
  fi
fi

wait_for_startup() {
  new_container_id=$(docker ps -f name=tines-app -q | head -n1)
  new_container_ip=$(docker inspect -f '{{range.NetworkSettings.Networks}}{{.IPAddress}}{{end}}' "$new_container_id")
  if ! "${docker_compose[@]}" run --rm tines-app curl --silent --retry-connrefused --retry 60 --retry-delay 1 --fail http://"$new_container_ip":3000/is_up; then
    log_cmd echo "Tines containers did not start successfully."
    exit 1
  fi
  log_cmd echo "Tines containers started successfully"
}

db_host=localhost
db_name=$(cat .env | grep DATABASE_NAME= | cut -d'=' -f2)
db_port=$(cat .env | grep DATABASE_PORT= | cut -d'=' -f2)
db_user=$(cat .env | grep DATABASE_USERNAME= | cut -d'=' -f2)

app_secret_token_placeholder="__SET_YOUR_SECRET_TOKEN__"
app_secret_token=$(openssl rand -hex 24)
log_cmd echo -e "\033[1m"
log_cmd echo -e "Generating app secret token"
log_cmd echo -e "\033[0m"

sed -i.backup -e "s/$app_secret_token_placeholder/$app_secret_token/" .env
if [ -f ".env" ]; then
  rm ".env.backup"
fi

log_cmd echo -e "\033[1m"
log_cmd echo -e "Generating database credential"
log_cmd echo -e "\033[0m"

DB_PLACEHOLDER="__SET_YOUR_DATABASE_PASSWORD__"
GENERATED_DB_PASSWORD=$(openssl rand -hex 12)
ENV_DB_PASSWORD=$(grep "^DATABASE_PASSWORD=" .env | awk -F '=' '{print $2}')

if [[ "$ENV_DB_PASSWORD" == "$DB_PLACEHOLDER" ]] && grep -q "$DB_PLACEHOLDER" docker-compose.yml; then
  # default database password is in use - generate new password and add it to both files

  log_cmd echo -e "\033[1m"
  log_cmd echo -e "Generating database credential"
  log_cmd echo -e "\033[0m"

  sed -i.backup -e "s/$DB_PLACEHOLDER/$GENERATED_DB_PASSWORD/" .env
  if [ -f ".env" ]; then
    rm ".env.backup"
  fi

  sed -i.backup -e "s/$DB_PLACEHOLDER/$GENERATED_DB_PASSWORD/" docker-compose.yml
  if [ -f "docker-compose.yml.backup" ]; then
    rm "docker-compose.yml.backup"
  fi

else
  if ! grep -q "$ENV_DB_PASSWORD" docker-compose.yml; then
    log_cmd echo "The DATABASE_PASSWORD variable in .env does not match the POSTGRES_PASSWORD variable in docker-compose.yml - exiting setup."
    exit 1
  fi
fi

################################################################################

log_cmd echo -e "\033[1m------------------------------------------------------------------"
log_cmd echo "WARNING: If database is not empty, all data will be lost"
log_cmd echo "------------------------------------------------------------------"
log_cmd echo -e "\033[0m"
read -p "Are you sure you want to continue? <Y/N> " prompt
if [[ $prompt != "y" && $prompt != "Y" && $prompt != "yes" && $prompt != "Yes" ]]; then
  echo
  exit 0
fi

log_cmd echo -e "\033[1m"
log_cmd echo -e "Loading Docker images"
log_cmd echo -e "\033[0m"
log_cmd docker load --input postgres14-5.tar
log_cmd docker load --input redis.tar
log_cmd docker load --input tines-app.tar
log_cmd docker load --input tines-nginx.tar
if [ -f tines-app.tar ]; then
  log_cmd docker load --input tines-app.tar
fi

if [ -f tines-app-fips.tar ]; then
  log_cmd docker load --input tines-app-fips.tar

  version=$(grep 'image: tines/tines-app' docker-compose.yml | head -n 1 | awk -F':' '{print $3}')
  if [ -n "$version" ]; then
    # If version is found, alias the fips image accordingly to tines-app so it continues to work during upgrades seamlessly
    docker tag tines/tines-app-fips:latest tines/tines-app:"$version"
  else
    # If no specific version is found, fallback to 'latest'
    docker tag tines/tines-app-fips:latest tines/tines-app:latest
  fi
fi

if [ -f tines-command-runner.tar ]; then
  log_cmd docker load --input tines-command-runner.tar
fi

if [ -f tines-ruby.tar ]; then
  log_cmd docker load --input tines-ruby.tar
fi
log_cmd echo -e "\033[1m"
log_cmd echo "Starting PostgreSQL and Redis first time"

setup_linux_user=$(grep -q '^[^#]*SETUP_DEDICATED_LINUX_USER=true' .env && echo "true" || echo "false")
set +e
if [[ $setup_linux_user == "true" ]]; then
  if setup_tines_user; then
    log_cmd echo -e "\033[1m"
    log_cmd echo -e "Tines user installed successfully"
    log_cmd echo -e "\033[0m"
  else
    log_cmd echo -e "\033[1m------------------------------------------------------------------"
    log_cmd echo "WARNING: Setting up tines user failed. Continuing with rest of the process. You may need to ensure that file permissions are correctly setup. "
    log_cmd echo "------------------------------------------------------------------"
    log_cmd echo -e "\033[0m"
    unset CONDITIONAL_LOCAL_TINES_USER
  fi
fi
set -e

log_cmd echo -e "\033[0m"
log_cmd "${docker_compose[@]}" up -d db redis

log_cmd echo "Waiting for Postgres to start..."
while ! nc -z $db_host "$db_port"; do sleep 0.1; done
log_cmd echo "Connected"

sleep 3

log_cmd echo -e "\033[1m"
log_cmd echo "Starting DB setup"
log_cmd echo -e "\033[0m"
log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake tines:ensure_db_created --trace
log_cmd echo
log_cmd echo "All available databases:"
log_cmd echo
log_cmd docker exec postgres psql -U "$db_user" -d "$db_name" -tAc "SELECT * FROM pg_database"

log_cmd echo -e "\033[1m"
log_cmd echo "Starting all Tines services"
log_cmd echo -e "\033[0m"
log_cmd "${docker_compose[@]}" up -d --timeout 30
log_cmd echo
log_cmd echo "Waiting for Tines containers to start..."
log_cmd echo
wait_for_startup
log_cmd echo
log_cmd echo "Docker containers:"
log_cmd docker ps -a

log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake tines:sync_integrations\["integrations"\]
log_cmd echo
log_cmd echo "Syncing public integrations."
log_cmd echo

log_cmd echo -e "\033[1m"
log_cmd echo "Tines application is ready to use."
log_cmd echo -e "\033[0m"
