#!/bin/bash

# shellcheck disable=SC2181,SC2086,SC2016

# Get the directory of the current script.
script_dir=$(dirname "$0")

# Change working directory to the script's directory.
cd "$script_dir" || exit
# shellcheck source=/dev/null
source "$script_dir/.common.sh"

resolve_docker_compose_command

# Check if Docker is installed.
if command -v docker &> /dev/null; then
  print_success_message "Docker is installed."
  append_to_result_log "container" "docker_installed" true
else
  print_error_message "Docker is not installed. Please install Docker to proceed." >> errors.log
  append_to_result_log "container" "docker_installed" false
fi

# Check if Docker Compose is installed.
if command -v $docker_compose &> /dev/null; then
  print_success_message "Docker Compose is installed."
  append_to_result_log "container" "docker_compose_installed" true
else
  print_error_message "Docker Compose is not installed. Please install Docker Compose to proceed." >> errors.log
  append_to_result_log "container" "docker_compose_installed" false
fi

# Print list of running containers.
print_message "Running containers:"
docker ps --format "table {{.ID}}\t{{.Names}}\t{{.Image}}\t{{.Ports}}"

# Check if application container can reach Postgres and Redis.
check_postgres_and_redis_connection=$($docker_compose run --rm tines-app bundle exec rake tines:startup_health_check)
# Check the exit status of the rake command.
if [ $? -eq 0 ]; then
  print_success_message "Application container can reach Postgres and Redis."
  print_message "$check_postgres_and_redis_connection"
  append_to_result_log "container" "can_connect_to_postgres_and_redis" true
else
  print_error_message "Application container encountered an error when verifying Postgres and Redis connections." >> errors.log
  append_to_result_log "container" "can_connect_to_postgres_and_redis" false
fi

# Check if nc (netcat) is available in the tines-app container.
$docker_compose run --rm tines-app bash -c 'command -v nc &> /dev/null'
if [ $? -eq 0 ]; then
  # If nc is available, check the SMTP connection.
  nc_output=$($docker_compose run --rm tines-app bash -c 'nc -v -d -q 3 -w 3 $SMTP_SERVER $SMTP_PORT')
  append_to_result_log "container" "netcat_available" true
  if [ $? -eq 0 ]; then
    print_success_message "Connection to SMTP server established successfully."
    print_message "$nc_output"
    append_to_result_log "container" "can_connect_to_smtp_server" true
    $docker_compose run --rm tines-app bundle exec rake tines:test_smtp_connection --trace
    if [ $? -eq 0 ]; then
      print_success_message "Successfully authenticated with SMTP server."
      append_to_result_log "container" "can_authenticate_with_smtp_server" true
    else
      print_error_message "Failed to authenticate with SMPT server." >> errors.log
      append_to_result_log "container" "can_authenticate_with_smtp_server" false
    fi
  else
    print_error_message "Connection to SMTP server failed." >> errors.log
    append_to_result_log "container" "can_connect_to_smtp_server" false
  fi
else
  print_error_message "nc (netcat) command not found in the tines-app container. Please install netcat in the container to perform SMTP checks." >> errors.log
  append_to_result_log "container" "netcat_available" false
fi
