#!/bin/bash

# shellcheck disable=SC2181

# Get the directory of the current script.
script_dir=$(dirname "$0")

# Change working directory to the script's directory.
cd "$script_dir" || exit
# shellcheck source=/dev/null
source "$script_dir/.common.sh"

resolve_docker_compose_command

# Run Rails console in the container to check database migrations status.
output=$($docker_compose run --rm tines-app bundle exec rails db:migrate:status)

# Check if any migrations are pending (i.e., status includes 'down' migrations).
if echo "$output" | grep -q "^\s*down"; then
  print_error_message "There are pending database migrations." >> errors.log
  print_message "$output" | grep -E "^\s*down\s+[0-9]{14}.*" >> errors.log
  append_to_result_log "database" "db_migrations_up_to_date" false
else
  # Check if the output contains any successful migration logs.
  if echo "$output" | grep -q "up\s*[0-9]\{14\}.*"; then
    print_success_message "All database migrations are up-to-date."
    append_to_result_log "database" "db_migrations_up_to_date" true
  else
    print_error_message "No successful database migrations found." >> errors.log
    print_message "Migrations status:"
    print_message "$output" >> errors.log
    append_to_result_log "database" "db_migrations_up_to_date" false
  fi
fi

# Get the used disk space percentage.
used_space_percentage=$(df -h / | awk 'NR==2 {print $5}' | tr -d '%')

# Check if less than 20% of disk space is available.
if [ "$used_space_percentage" -gt 80 ]; then
  print_error_message "WARNING: Low remaining disk space (${used_space_percentage}% used)."
  print_error_message "Consider deleting older database backups to free up space."
  append_to_result_log "database" "passed_disk_space_check" false
else
  append_to_result_log "database" "passed_disk_space_check" true
fi
append_to_result_log "database" "used_disk_space_percentage" "$used_space_percentage%"

# Print tenant info.
$docker_compose run --rm tines-app bundle exec rake tines:print_tenant_info
if [ $? -ne 0 ]; then
  print_error_message "Error retrieving tenant info." >> errors.log
fi
