#!/bin/bash

# Check if the required environment variables are present for tenant, email, database, and core configuration.

# Check if the script is executed within the container.
if [ -z "$RUNNING_IN_CONTAINER" ]; then
  # Get the directory of the current script.
  script_dir=$(dirname "$0")

  # Change working directory to the script's directory.
  cd "$script_dir" || exit
  # shellcheck source=/dev/null
  source "$script_dir/.common.sh"

  resolve_docker_compose_command

  $docker_compose run --volume "/opt/tines/":/tmp --rm tines-app bash -c "RUNNING_IN_CONTAINER=1 /tmp/troubleshoot/environment-variables.sh"
else
  # The script is being executed within the container.

  # Exit early if .env.tmpl is not present.
  env_file="/tmp/.env.tmpl"
  if [ ! -f "$env_file" ]; then
    echo -e "❌  $env_file not found."
    exit 1
  fi

  # Define .env.tmpl sections.
  current_section=""
  core_config_section="Core configuration"
  optional_config_section="Optional feature configuration"

  # Iterate through .env.tmpl line by line.
  while IFS= read -r line; do
    # Determine what section of .env.tmpl file we're in.
    if [[ $line == "# Required:"* ]]; then
      current_section=$(echo "$line" | sed -E 's/# Required: (.*)/\1/' | tr -d '#')
      continue
    elif [[ $line == "# Core configuration"* ]]; then
      current_section=$(echo "$line" | sed -E 's/# (.*)/\1/')
      continue
    elif [[ $line == "# Optional feature configuration"* ]]; then
      current_section=$(echo "$line" | sed -E 's/# (.*)/\1/' | tr -d '#' | sed -E 's/[[:space:]]*$//')
      continue
    elif [[ $line == "#"* ]] || [[ "$line" =~ ^[[:space:]]*$ ]]; then
      continue # Skip blank and commented lines.
    fi

    # Extract the environment variable's name and value.
    var_name=$(echo "$line" | awk -F '=' '{print $1}' | tr -d '[:space:]')
    var_value=$(echo "$line" | awk -F '=' '{print $2}' | tr -d '[:space:]')

    # Validate the current environment variable is set.
    if [ -z "${!var_name}" ]; then
      # Skip missing optional feature configuration.
      if [ "$current_section" == "$optional_config_section" ]; then
        continue
      else
        echo -e "❌  Environment variable $var_name is not set or is empty."
      fi
    # Validate core environment variable's values.
    elif [ "$current_section" == "$core_config_section" ]; then
      if [ "${!var_name}" != "$var_value" ]; then
        echo -e "❌  $var_name should be set to \'$var_value\'."
      fi
    fi
  done < "$env_file"
fi
