#!/bin/bash

########################################################################################
# Troubleshooting script for Docker Compose installations of Tines
########################################################################################
# This bash script is designed to assist customers in identifying and resolving common
# issues that may arise during the installation and usage of Tines with docker-compose.
# The script performs various checks and diagnostic steps to help diagnose potential
# problems related to the environment, configuration, and dependencies.
#
# Usage: run.sh
#
# Individual troubleshooting sections may also be run.
# For example, to verify environment variables only, run: environment-variables.sh
#
#
# IMPORTANT: Please ensure you have docker-compose installed and the necessary
#            permissions to run Docker commands before executing this script.
#
# Note: This script is not a comprehensive solution for all possible issues, but it can
#       provide valuable information to support the troubleshooting process. If the
#       problem persists or is not covered by this script, please refer to the official
#       documentation or contact our support team for further assistance.
########################################################################################

# Get the directory of the current script.
script_dir=$(dirname "$0")

# Change working directory to the script's directory.
cd "$script_dir" || exit
# shellcheck source=/dev/null
source "$script_dir/.common.sh"

rm -f "$script_dir/result.log"
rm -f "$script_dir/errors.log"
rm -f "$script_dir/env-var-errors.log"

print_heading "Checking environment variables..."
# shellcheck source=/dev/null
source "$script_dir/environment-variables.sh" >> env-var-errors.log
if [ ! -s env-var-errors.log ]; then
  print_success_message "Environment variables checks passed."
else
  print_error_message "Environment variables checks failed (see end of logs for details)."
fi

print_heading "Checking connection to hq.tines.io..."
# shellcheck source=/dev/null
source "$script_dir/hq.sh"

print_heading "Checking container..."
# shellcheck source=/dev/null
source "$script_dir/container.sh"

print_heading "Checking database..."
# shellcheck source=/dev/null
source "$script_dir/database.sh"

# Print the errors, if any, at the end of the script.
if [ -s errors.log ] || [ -s env-var-errors.log ]; then
  print_heading "Errors"
  print_message "The following errors were found:"
  cat "$script_dir/env-var-errors.log"
  cat "$script_dir/errors.log"
  exit 1
else
  print_heading "Success"
  print_success_message "All checks passed. No errors found."
fi
