#! /bin/bash

set -e

LOG_DIR="$(pwd)/log/upgrade-logs"
LOG_FILE="$LOG_DIR/upgrade_$(date +%Y-%m-%d_%H-%M-%S).log"
mkdir -p "$LOG_DIR"

log_cmd() {
  "$@" 2>&1 | tee -a "$LOG_FILE"
}
exit_with_error_message() {
  log_cmd echo "$1"
  exit 1
}
cleanup_old_logs() {
  cd "$LOG_DIR"
  ls -1tr | head -n -10 | xargs -d '\n' rm -f --
  cd - > /dev/null
}
cleanup_old_logs

# Volume name pattern is, for example, <directory_name>_db-data
DIRECTORY="$(basename "$(pwd)")_db-data"
VOLUME="$(docker volume ls -f name="$DIRECTORY" | tail -n 1 | awk '{print $2}')"

if [[ "$DIRECTORY" != "$VOLUME" ]]; then
  exit_with_error_message "Volume name pattern does not match directory name. Please ensure you are running this script from the directory you set up Tines in and that the directory name has not been changed."
fi

if docker compose version &> /dev/null; then
  docker_compose=(docker compose)
else
  docker_compose=(docker-compose)
fi

if [[ "${USE_OLD_DOCKER_COMPOSE_SHELL_COMMAND:-false}" == "true" ]]; then
  log_cmd echo "Using docker-compose as shell command"
  docker_compose=(docker-compose)
fi

log_cmd echo "===== Docker status:"
log_cmd docker -v
log_cmd "${docker_compose[@]}" version
log_cmd echo
log_cmd echo "===== Docker containers:"
log_cmd docker ps -a
log_cmd echo "===== Docker networks:"
log_cmd docker network ls
log_cmd echo
log_cmd echo "===== Docker volumes:"
log_cmd docker volume ls
log_cmd echo
log_cmd echo "Current directory: $(pwd)"
log_cmd echo

wait_for_pg() {
  n=0
  until [ "$n" -ge 10 ]; do
    log_cmd "${docker_compose[@]}" exec -T db psql -U tines -c "SELECT 1 as result_one" | grep -q "result_one" && break
    n=$((n + 1))
    sleep 5
  done

  if [ "$n" -ge 10 ]; then
    log_cmd "${docker_compose[@]}" logs db
    exit_with_error_message "Postgres container failed to come up. Please try again or reach out to support."
  fi
}

if [ ! -f "docker-compose.yml" ] && [ ! -f "docker-compose.yaml" ]; then
  exit_with_error_message "No docker-compose.yml file found in this directory. You'll need to run this script from the directory you extracted the release ZIP file to."
fi

if [ ! -f ".env" ]; then
  exit_with_error_message "No .env file found in this directory. Are you running this script in the directory that you're currently running Tines from?"
fi

OLD_PASSWORD=$(cat .env | grep APP_SECRET_TOKEN= | cut -d'=' -f2)
OLD_PASSWORD_HASH="5392a99c55fd5a7257952b63f41302b2d7137b7c2cb8356c6feee76b41f91ce6"
APP_SECRET_TOKEN_HASH=$(echo -n "$OLD_PASSWORD" | openssl dgst -sha256)
NEW_PASSWORD=$(openssl rand -hex 24)

if [[ "$OLD_PASSWORD_HASH" == "$APP_SECRET_TOKEN_HASH" ]]; then
  log_cmd echo "===== Found old APP_SECRET_TOKEN. Rotating now"

  sed -i.backup -e "s/$OLD_PASSWORD/$NEW_PASSWORD/" .env
  if [ -f ".env" ]; then
    rm ".env.backup"
  fi
fi

reload_nginx() {
  log_cmd docker exec nginx /usr/sbin/nginx -s reload
}

if ! docker info > /dev/null 2>&1; then
  log_cmd echo "Docker Compose services aren't running. In order to proceed we will need to ensure that Docker is running"
  exit
fi

setup_linux_user=$(grep -q '^[^#]*SETUP_DEDICATED_LINUX_USER=true' .env && echo "true" || echo "false")
if [[ $setup_linux_user == "true" ]]; then
  export CONDITIONAL_LOCAL_TINES_USER="8463:8463"
  export CONDITIONAL_MAPPED_RO_LOCAL_ETC_PASSWD_FILE="/etc/passwd:/etc/passwd:ro"
  export CONDITIONAL_MAPPED_RO_LOCAL_ETC_GROUP_FILE="/etc/group:/etc/group:ro"
fi

# Check for expected postgres version
if ! "${docker_compose[@]}" ps --services --filter "status=running" | grep -q db; then
  log_cmd echo "Postgres container is not running. Starting only the postgres container temporarily."

  log_cmd "${docker_compose[@]}" up -d db
  wait_for_pg
fi
log_cmd echo "All available Docker volumes for Postgres DB:"
log_cmd docker volume ls --format "{{.Name}}"
tines_postgres=$(docker inspect postgres --format "{{range (.Mounts)}}{{println .Name}}{{end}}" | grep data)
log_cmd echo "Docker volume name for Postgres: $tines_postgres"

log_cmd echo "===== Stop all Tines containers"
log_cmd echo
log_cmd "${docker_compose[@]}" stop
log_cmd echo

log_cmd echo "===== Backup Postgres database"
log_cmd echo

today=$(date +"%Y-%m-%d")
backup_dir="db-backup"

# KEEP_LAST_DATABAS_BACKUPS is deprecated
backup_limit=${KEEP_LAST_DATABASE_BACKUPS:-${KEEP_LAST_DATABAS_BACKUPS:-10}}

mkdir -p "$backup_dir/$today"
cp -r /var/lib/docker/volumes/"$tines_postgres"/_data/ ./"$backup_dir/$today"/ || exit_with_error_message "Failed to backup Postgres data - cancelling upgrade."

num_backups=$(find "$backup_dir" -maxdepth 1 -type d | wc -l)

if [ "$num_backups" -gt "$backup_limit" ]; then
  num_to_delete=$((num_backups - backup_limit))
  log_cmd echo "Keeping last $backup_limit backups. Deleting $num_to_delete old backups:"

  old_backups=$(find "$backup_dir" -maxdepth 1 -type d | grep -v "$backup_dir/$today" | sort | head -n "$num_to_delete")
  if echo "$old_backups" | grep -q "$backup_dir/$today"; then
    exit_with_error_message "Error: Attempting to delete the most recent backup. Aborting."
  fi

  echo "$old_backups" | while IFS= read -r dir; do
    log_cmd echo "Cleaning up old backup directories. Deleting: $dir"
    rm -rf "$dir"
  done
fi

log_cmd echo

log_cmd echo "===== Backup .env file"
timestamp=$(date +%s)
cp .env .env.old."$timestamp"
log_cmd echo

log_cmd echo "===== Load new Tines images"
log_cmd docker load --input postgres14-5.tar
log_cmd docker load --input redis.tar
log_cmd docker load --input tines-app.tar

if [ -f tines-app.tar ]; then
  log_cmd docker load --input tines-app.tar
fi

if [ -f tines-app-fips.tar ]; then
  log_cmd docker load --input tines-app-fips.tar

  version=$(grep 'image: tines/tines-app' docker-compose.yml | head -n 1 | awk -F':' '{print $3}')
  if [ -n "$version" ]; then
    # If version is found, alias the fips image accordingly to tines-app so it continues to work during upgrades seamlessly
    docker tag tines/tines-app-fips:latest tines/tines-app:"$version"
  else
    # If no specific version is found, fallback to 'latest'
    docker tag tines/tines-app-fips:latest tines/tines-app:latest
  fi
fi

if [ -f tines-command-runner.tar ]; then
  log_cmd docker load --input tines-command-runner.tar
fi

if [ -f tines-ruby.tar ]; then
  log_cmd docker load --input tines-ruby.tar
fi

log_cmd docker load --input tines-nginx.tar
log_cmd echo

log_cmd echo "===== Update Redis and PostgreSQL containers"
log_cmd "${docker_compose[@]}" up -d db redis

log_cmd echo "===== Apply database changes"
log_cmd echo
log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake db:safe_migrate --trace || exit_with_error_message "Failed to apply database changes - cancelling upgrade."
log_cmd echo

log_cmd echo "===== Bootstrap partitioned database tables"
log_cmd echo
log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake db:bootstrap_partitioned_tables --trace || exit_with_error_message "Failed to bootstrap partitioned database tables - cancelling upgrade."
log_cmd echo

log_cmd echo "===== Sync tenant object with .env file if tenant exists"
log_cmd echo
log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake tines:update_tenant_from_env || echo "Failed to sync tenant data from .env - continuing with upgrade."
log_cmd echo

log_cmd echo "===== Syncing public integrations"
log_cmd "${docker_compose[@]}" run --rm tines-app bundle exec rake tines:sync_integrations\["integrations"\] || echo "Failed to sync tenant data from .env - continuing with upgrade."
log_cmd echo

log_cmd echo "===== Start new Tines containers"
log_cmd echo
log_cmd "${docker_compose[@]}" up --timeout 30 -d --remove-orphans
log_cmd echo

log_cmd echo "===== Remove old Tines images"
log_cmd echo
OLD_IMAGES=$(docker images -f "dangling=true" -q)
[[ $OLD_IMAGES ]] && echo "$OLD_IMAGES" | xargs docker rmi
log_cmd echo

log_cmd echo "===== Docker containers:"
log_cmd echo
log_cmd docker ps -a
log_cmd echo

log_cmd echo "===== Update finished"
